********************************************************************************
* Replication code for Table A1 "Unusual income and financial literacy"
*
* Title: Consumer Credit with Over-Optimistic Borrowers 
*
* Authors: Florian Exler, Igor Livshits, James MacGee and Michèle Tertilt
*
* Date: 23rd July, 2024
********************************************************************************


		***************
		* Description *
		***************
		
/* 
	Input(s): Two main survey data files from Survey of Consumer Finances: waves 2019 and 2016
	
	Output: Latex code "table_A1.tex" for Table A1 

*/

		**********************
		* Clear and set path *
		**********************
		
		
clear all
cls
set maxvar 5500

args inputfolderpath outfolderpath

// Print the arguments to verify they are received correctly
display "inputfolderpath: `inputfolderpath'"
display "outfolderpath: `outfolderpath'"

		*******************************************
		* Load and append SCF data from 2016-2019 *
		*******************************************

		
* Load SCF 2019
u "`inputfolderpath'/p19i6.dta", clear 
keep y1 x42001 x7650 x7558 x7559 x7560 x14 x5931 x5729
g wave=2019

* Load and append SCF 2016
append using "`inputfolderpath'/p16i6.dta", keep (y1 x42001 x7650 x7558 x7559 x7560 x14 x5931 x5729)
replace wave=2016 if wave==.

* Drop replicated observations from imputations 
g id = y1
keep if mod(id,10) ==1 // nb of obs should be 31,240/5 = 6248



		*********************************
		* Construct variables for table *
		*********************************

		
* Statistical weights and age
g weights=x42001 
g age = x14

* Total Income 
g income_total=max(0,x5729)

* Whether income unusually low/high or normal 
g income_higher_usual = 0
replace income_higher_usual =1 if x7650==1

g income_lower_usual = 0
replace income_lower_usual =1 if x7650==2

g income_usual = 0
replace income_usual =1 if x7650==3


* Financial literacy questions
g fin_lit_q1 =0
replace fin_lit_q1=1 if x7558 == 5

g fin_lit_q2 =0
replace fin_lit_q2=1 if x7559 == 1

g fin_lit_q3 =0
replace fin_lit_q3 =1 if x7560==5


* Number of correctly answered financial literacy questions: 
g fin_lit_score = fin_lit_q1+fin_lit_q2+fin_lit_q3

** 0 or 1 correct questions: 
g fin_lit_0_1 = 0
replace fin_lit_0_1 =1 if fin_lit_score < 2

** 2 or 3 correct questions: 
g fin_lit_2_3 = 0
replace fin_lit_2_3 =1 if fin_lit_score > 1
 

* Education college dummy == 1 if max(x5931,0) >= 10 
g college_dummy =0
replace college_dummy =1 if  max(x5931,0) >= 10 


		*****************************
		* Sample Restrictions (Age) *
		*****************************

global age_min=25
global age_max =55 
keep if (age>=$age_min)&(age<=$age_max) 


		*******************
		* Create Table A1 *
		*******************

capture file close myfile

* No college degree

** Share of correct questions: 

su fin_lit_0_1 if (college_dummy==0) [iw=weights]  
local  share_corr_q_0_1 = "`=string(`r(mean)', "%9.2f")'"

su fin_lit_2_3 if (college_dummy==0) [iw=weights]  
local  share_corr_q_2_3 = "`=string(`r(mean)', "%9.2f")'"


** Fraction with income unusually low - 0/1 correct questions
su income_lower_usual if (fin_lit_0_1==1)&(college_dummy==0) [iw=weights]  
local  share_inc_low_corr_q_0_1 = "`=string(`r(mean)', "%9.2f")'"
g ratio = r(mean)


** Fraction with income unusually low - 2/3 correct questions
su income_lower_usual if(fin_lit_2_3==1)&(college_dummy==0)  [iw=weights] 
local  share_inc_low_corr_q_2_3 = "`=string(`r(mean)', "%9.2f")'"

** Fraction with income unusually low - Ratio 0/1 over 2/3 correct questions
replace ratio = ratio/r(mean)
su ratio
local  share_inc_low_ratio = "`=string(`r(mean)', "%9.2f")'"
  
drop ratio
** Fraction with normal income  - 0/1 correct questions
su income_usual if (fin_lit_0_1==1)&(college_dummy==0) [iw=weights] 
local  share_inc_norm_corr_q_0_1 = "`=string(`r(mean)', "%9.2f")'"
di "`share_inc_norm_corr_q_0_1'"

g ratio = r(mean)

** Fraction with normal income - 2/3 correct questions
su income_usual if(fin_lit_2_3==1)&(college_dummy==0)  [iw=weights] 
local  share_inc_norm_corr_q_2_3 = "`=string(`r(mean)', "%9.2f")'"

** Fraction with normal income - Ratio 0/1 over 2/3 correct questions
replace ratio = ratio/r(mean)
su ratio
local  share_inc_norm_ratio = "`=string(`r(mean)', "%9.2f")'"


drop ratio

** Fraction with income unusually high - 0/1 correct questions
su income_higher_usual if (fin_lit_0_1==1)&(college_dummy==0) [iw=weights] 
local  share_inc_high_corr_q_0_1 = "`=string(`r(mean)', "%9.2f")'"

g ratio = r(mean)

** Fraction with income unusually high - 2/3 correct questions
su income_higher_usual if(fin_lit_2_3==1)&(college_dummy==0)  [iw=weights] 
local  share_inc_high_corr_q_2_3 = "`=string(`r(mean)', "%9.2f")'"

** Fraction with high income - Ratio 0/1 over 2/3 correct questions
replace ratio = ratio/r(mean)
su ratio
local  share_inc_high_ratio = "`=string(`r(mean)', "%9.2f")'"

drop ratio

** Mean total income - 0/1 correct questions
su income_total if (fin_lit_0_1==1)&(college_dummy==0) [iw=weights] 
local  total_inc_0_1 = round(r(mean))

** Mean total income - 2/3 correct questions
su income_total if(fin_lit_2_3==1)&(college_dummy==0)  [iw=weights] 
local  total_inc_2_3 = round(r(mean))


* With college degree

** Share of correct questions: 

su fin_lit_0_1 if (college_dummy==1) [iw=weights]  
local  share_corr_q_0_1_col = "`=string(`r(mean)', "%9.2f")'"

su fin_lit_2_3 if (college_dummy==1) [iw=weights]  
local  share_corr_q_2_3_col = "`=string(`r(mean)', "%9.2f")'"


** Fraction with income unusually low - 0/1 correct questions
su income_lower_usual if (fin_lit_0_1==1)&(college_dummy==1) [iw=weights]  
local share_inc_low_corr_q_0_1_col = "`=string(`r(mean)', "%9.2f")'"
g ratio = r(mean)


** Fraction with income unusually low - 2/3 correct questions
su income_lower_usual if(fin_lit_2_3==1)&(college_dummy==1)  [iw=weights] 
local  share_inc_low_corr_q_2_3_col = "`=string(`r(mean)', "%9.2f")'"

** Fraction with income unusually low - Ratio 0/1 over 2/3 correct questions
replace ratio = ratio/r(mean)
su ratio
local  share_inc_low_ratio_col = "`=string(`r(mean)', "%9.2f")'"
  
drop ratio
** Fraction with normal income  - 0/1 correct questions
su income_usual if (fin_lit_0_1==1)&(college_dummy==1) [iw=weights] 
local  share_inc_norm_corr_q_0_1_col = "`=string(`r(mean)', "%9.2f")'"

g ratio = r(mean)

** Fraction with normal income - 2/3 correct questions
su income_usual if(fin_lit_2_3==1)&(college_dummy==1)  [iw=weights] 
local  share_inc_norm_corr_q_2_3_col = "`=string(`r(mean)', "%9.2f")'"

** Fraction with normal income - Ratio 0/1 over 2/3 correct questions
replace ratio = ratio/r(mean)
su ratio
local  share_inc_norm_ratio_col = "`=string(`r(mean)', "%9.2f")'"


drop ratio

** Fraction with income unusually high - 0/1 correct questions
su income_higher_usual if (fin_lit_0_1==1)&(college_dummy==1) [iw=weights] 
local  share_inc_high_corr_q_0_1_col = "`=string(`r(mean)', "%9.2f")'"

g ratio = r(mean)

** Fraction with income unusually high - 2/3 correct questions
su income_higher_usual if(fin_lit_2_3==1)&(college_dummy==1)  [iw=weights] 
local  share_inc_high_corr_q_2_3_col = "`=string(`r(mean)', "%9.2f")'"

** Fraction with high income - Ratio 0/1 over 2/3 correct questions
replace ratio = ratio/r(mean)
su ratio
local  share_inc_high_ratio_col = "`=string(`r(mean)', "%9.2f")'"

drop ratio

** Mean total income - 0/1 correct questions
su income_total if (fin_lit_0_1==1)&(college_dummy==1) [iw=weights] 
local  total_inc_0_1_col = round(r(mean))

** Mean total income - 2/3 correct questions
su income_total if(fin_lit_2_3==1)&(college_dummy==1)  [iw=weights] 
local  total_inc_2_3_col = round(r(mean))


file open myfile using "`outfolderpath'/TableA1.tex", write text replace

file write myfile "\begin{table}[h!]"_n
file write myfile "\centering" _n
file write myfile "\caption{Unusual income and financial literacy}"_n
file write myfile "\begin{tabular}{lccccc}"_n
file write myfile "\hline"_n
file write myfile "\# Correct &  Share & \multicolumn{3}{c}{Fraction with income unusually}  & Total\\[-5ex]\\" _n
file write myfile "questions & & low & normal & high & income\\"

file write myfile "\hline"_n
file write myfile "\textbf{No college degree} & & & & & \\"_n
file write myfile "0 or 1 & `share_corr_q_0_1' & `share_inc_low_corr_q_0_1' & `share_inc_norm_corr_q_0_1' & `share_inc_high_corr_q_0_1' & `total_inc_0_1' \\"_n
file write myfile "2 or 3 & `share_corr_q_2_3' & `share_inc_low_corr_q_2_3' & `share_inc_norm_corr_q_2_3' & `share_inc_high_corr_q_2_3' & `total_inc_2_3' \\"_n
file write myfile "Ratio (0 or 1) / (2 or 3) & & `share_inc_low_ratio' & `share_inc_norm_ratio' & `share_inc_high_ratio'  & \\"_n

file write myfile "\hline"_n

file write myfile "\textbf{With college degree} & & & & & \\"_n
file write myfile "0 or 1 & `share_corr_q_0_1_col' & `share_inc_low_corr_q_0_1_col' & `share_inc_norm_corr_q_0_1_col' & `share_inc_high_corr_q_0_1_col' & `total_inc_0_1_col' \\"_n
file write myfile "2 or 3 & `share_corr_q_2_3_col' & `share_inc_low_corr_q_2_3_col' & `share_inc_norm_corr_q_2_3_col' & `share_inc_high_corr_q_2_3_col' & `total_inc_2_3_col' \\"_n
file write myfile "Ratio (0 or 1) / (2 or 3) & & `share_inc_low_ratio_col' & `share_inc_norm_ratio_col' & `share_inc_high_ratio_col'  & \\"_n

file write myfile "\hline"_n
file write myfile "\end{tabular}"_n
file write myfile "\caption*{Note: Results for pooled SCF 2016 and 2019 for 25-55 years olds. With college degree are households that report at least a first college degree for the household head (x5931). Total income is the total received income of the household from all sources before taxes and deductions (x5729).}"_n
file write myfile "\end{table}"_n

file close myfile

exit, clear
